PD equ 0x17
VER equ 0x07
; ------------------------------------------------------------------------
;
; Title:
;
;   PD17 -- PIC 1/2.5/5/10 MHz to 1 PPS frequency divider, with sync
;
; Function:
;
;   This PIC program implements a digital frequency divider: the external
;   1, 2.5, 5, or 10 MHz input clock is divided down to 1 Hz (1 pulse per
;   second). Two pins are used to configure the frequency division ratio.
;
;                               ---o---
;              (Vdd) +5 V  ++++|1     8|====  ground (Vss)
;          n MHz clock in  >---|2 12F 7|+--<  CfgA
;                1PPS out  <---|3 6xx 6|+--<  CfgB
;                     ARM  >--*|4     5|+--<  SYNC
;                               -------
; Notes:
;
;  - The 1PPS output has 10% duty cycle (100 ms pulse width).
;  - 1PPS rising edge can be synchronized using ARM-SYNC control pins.
;  - Output frequency accuracy is the same as clock input accuracy.
;  - Jitter is below 1 ps (the PIC is a fully synchronous device).
;  - Internal weak pull-up on pins 7,6,5; external pull-up on pin 4.
;  - Coded for PIC 12F675 but any '609 '615 '629 '635 '675 '683 works.
;
; Version:
;
;   30-Jul-2008  Tom Van Baak (tvb)  http://www.LeapSecond.com/pic
;
; ------------------------------------------------------------------------

; Define processor, fuses, register base, and interrupt vectors.

        list        p=pic12f675
        include     "p12f675.inc"
        __config    _EC_OSC & _MCLRE_OFF & _WDT_OFF & _PWRTE_ON
        cblock  0x20
        endc
        org     0               ; power-on entry
        goto    init
        dw      PD, VER
        org     4               ; (4) interrupt entry
        goto    sync            ; (2)

; One-time PIC 12F675 initialization.

init:   bcf     STATUS,RP0      ; bank 0
        clrf    GPIO            ; set outputs low
        movlw   0x07            ; turn comparator off
        movwf   CMCON           ;

        bsf     STATUS,RP0      ; bank 1
        errorlevel -302
        clrf    ANSEL           ; all digital (no analog) pins
        movlw   ~(1<<GP4)
        movwf   TRISIO          ; set pin directions (0=output)
        movlw   1<<GP0 | 1<<GP1 | 1<<GP2
        movwf   WPU             ; enable weak pullup (1=enable)
        movlw   0<<NOT_GPPU | 1<<INTEDG
        movwf   OPTION_REG      ; WPU, GP2/INT rising edge trigger
        errorlevel +302
        bcf     STATUS,RP0      ; bank 0
        call    conf            ; set user-defined clock rate

; Generate 10% duty cycle pulse inside an isochronous 1 Hz loop.
; - Pulse width and period are cycle accurate.
; - Continues forever, unless ARM pin goes low.

        ; Set output high for exactly 100 ms.

loop:   bsf     GPIO,GP4        ; (1) 1PPS rise
        call    Delay5          ; (5)
sync:   goto    $+1             ; (2)
        nop                     ; (1)
        movlw   d'9'            ; (1)
        call    Delay_W_10ms    ; (0) + 90 ms
        call    Delay_10ms_10cy ; (-10) + 10 ms

        ; Set output low for exactly 900 ms.

        bcf     GPIO,GP4        ; (1) 1PPS fall
        movlw   d'89'           ; (1)
        call    Delay_W_10ms    ; (0) + 890 ms
        call    Delay_10ms_10cy ; (-10) + 10 ms
        call    Delay5          ; (5)
        btfsc   GPIO,GP3        ; (1) check ARM pin
          goto  loop            ; (1/2) high=run, low=arm

; Implement ARM-SYNC protocol using GP2/INT-interrupt.
; - Hold ARM pin low for a second or more to stop the divider.
; - The 1PPS output will synchronize to rising edge of SYNC pin.
; - Synchronization accuracy is 1 PIC cycle (4 clock cycles).

arm:    call    conf            ; [re]set user-defined clock rate
        bsf     GPIO,GP4        ; set 1PPS high
        movlw   1<<GIE | 1<<INTE
        movwf   INTCON          ; enable GP2 edge-trigger interrupt
        goto    $               ; no deposit, no return, no retfie

; Read configuration pins (WPU) to set nominal clock rate.
;   00 =  1.0 MHz (ground both)
;   01 =  2.5 MHz (ground CfgB)
;   10 =  5.0 MHz (ground CfgA)
;   11 = 10.0 MHz (default)

conf:   clrf    clkcfg          ; clear 2-bit (4-state) frequency mode
        btfsc   GPIO,GP0        ; if config A is high,
          bsf   clkcfg,0        ;   add 1
        btfsc   GPIO,GP1        ; if config B is high,
          bsf   clkcfg,1        ;   add 2
        return                  ;

        include "delayw.asm"
        include "delay10ms.asm"
        end

